/************************************************************************/
/*                                                                      */
/* Borland Enterprise Core Objects                                      */
/*                                                                      */
/* Copyright (c) 2003-2005 Borland Software Corporation                 */
/*                                                                      */
/************************************************************************/

using System;
using System.ComponentModel;
using System.ComponentModel.Design;
using System.Drawing.Design;

using Borland.Eco.ObjectRepresentation;
using Borland.Eco.Subscription;
using Borland.Eco.UmlRt;
using Borland.Eco.Services;
using Borland.Eco.Support.Deriver;
using Borland.Eco.Handles;
using Borland.Eco.Exceptions;
using Borland.Eco.Globalization;

namespace Borland.Eco.Handles
{

	internal sealed class RootHandleConverter: ReferenceConverter
	{
		public RootHandleConverter(Type type): base(type) {}

		///<exception cref="ArgumentNullException">Thrown if <paramref name="context"/> is null.</exception>
		protected override bool IsValueAllowed(ITypeDescriptorContext context, object value)
		{
			if (context == null) throw new ArgumentNullException("context"); // do not localize
			ElementHandle currentAsHandle = null;
#if !CF
			currentAsHandle = context.Instance as ElementHandle;
#endif
			if (currentAsHandle == null)
				return true;
			if (currentAsHandle.Equals(value))
				return false;
			RootedHandle valueAsRootedhandle = value as RootedHandle;
			if (valueAsRootedhandle != null)
				return !(valueAsRootedhandle).IsRootLinkedTo(currentAsHandle);
			return true;
		}
	}

	[Designer("Borland.Eco.Handles.Design.RootedHandleDesigner, Borland.Eco.Handles.Design", typeof(IDesigner))]
	public abstract class RootedHandle: ElementHandle
	{
		private sealed class ActiveChangedAdapter: SubscriberAdapterBase
		{
			public ActiveChangedAdapter(object subscriber): base(subscriber) 
			{
			}
			protected override void DoReceive(object sender, EventArgs e, object actualSubscriber)
			{
				if (actualSubscriber == null) throw new ArgumentNullException("actualSubscriber"); // do not localize
				(actualSubscriber as RootedHandle).ActiveChanged();
			}
		}
		private sealed class RootElementChangedAdapter: SubscriberAdapterBase
		{
			public RootElementChangedAdapter(object subscriber): base(subscriber) 
			{
			}
			protected override void DoReceive(object sender, EventArgs e, object actualSubscriber)
			{
				if (actualSubscriber == null) throw new ArgumentNullException("actualSubscriber"); // do not localize
				(actualSubscriber as RootedHandle).EffectiveRootValueChanged();
			}			
		}

		private sealed class RootStaticContextChangedAdapter: SubscriberAdapterBase
		{
			public RootStaticContextChangedAdapter(object subscriber): base(subscriber) 
			{
			}
			protected override void DoReceive(object sender, EventArgs e, object actualSubscriber)
			{
				if (actualSubscriber == null) throw new ArgumentNullException("actualSubscriber"); // do not localize
				(actualSubscriber as RootedHandle).RootStaticContextChanged();
			}			
		}

		private sealed class RootedHandleDeriver: AbstractDeriver
		{
			private RootedHandle rootedHandle;

			public RootedHandleDeriver(RootedHandle OwningRootedHandle): base()
			{
				rootedHandle = OwningRootedHandle;
			}
			protected override void DoNotifyOutOfDate()
			{
				rootedHandle.ElementChanged();
			}

			protected override void DoDeriveAndSubscribe(bool subscribe)
			{
				if (subscribe)
					rootedHandle.DeriveAndSubscribe(ReevaluateSubscriber,ResubscribeSubscriber);
				else
					rootedHandle.DeriveAndSubscribe(null, null);
			}
		}

		private sealed class VariableAdder: object, IStaticContext
		{
			private RootedHandle owner;

			private IStaticContext RootContext
			{
				get { return owner.RootHandle; }
			}

			public VariableAdder(RootedHandle owner)
			{
				this.owner = owner;
			}

			IEcoTypeSystem IStaticContext.TypeSystem
			{
				get
				{
					if (RootContext != null)
						return RootContext.TypeSystem;
					else
						return null;
				}
			}

			IOclTypeService IStaticContext.OclTypeService
			{
				get
				{
					if (RootContext != null)
						return RootContext.OclTypeService;
					else
						return null;
				}
			}

			IOclPsTypeService IStaticContext.OclPsTypeService
			{
				get
				{
					if (RootContext != null)
						return RootContext.OclPsTypeService;
					else
						return null;
				}
			}

			IActionLanguageTypeService IStaticContext.ActionLanguageTypeService
			{
				get
				{
					if (RootContext != null)
						return RootContext.ActionLanguageTypeService;
					else
						return null;
				}
			}
			IExternalVariableList IStaticContext.VariableList
			{
				get { return owner.EffectiveVariables(); }
			}
			IClassifier IStaticContext.StaticUmlType
			{
				get
				{
					if (RootContext != null)
						return RootContext.StaticUmlType;
					else
						return null;
				}
			}
		}

		private AbstractDeriver deriver;

		protected sealed override IOclTypeService GetOclTypeService()
		{
			return EffectiveRootStaticContext.OclTypeService;
		}

		protected sealed override IOclPsTypeService GetOclPsTypeService()
		{
			return EffectiveRootStaticContext.OclPsTypeService;
		}

		protected sealed override IActionLanguageTypeService GetActionLanguageTypeService()
		{
			return EffectiveRootStaticContext.ActionLanguageTypeService;
		}

		protected sealed override object GetEcoService(System.Type serviceType)
		{
			if (Active) // will check for RootHandle != null
			{
				IEcoServiceProvider esp = RootHandle;
				return esp.GetEcoService(serviceType);
			}
			else
				return null;
		}

		private ElementHandle rootHandle;
		private RootElementChangedAdapter rootChangedAdapter;
		private RootStaticContextChangedAdapter rootStaticContextChangedAdapter;

		internal void EffectiveRootValueChanged()
		{
			MarkSubscriptionOutOfDate();
		}

		internal void RootStaticContextChanged()
		{
			StaticContextChanged();
			EffectiveRootValueChanged();
			effectiveVariables = null;
		}

		protected IElement EffectiveRootValue()
		{
			if (RootHandle != null)
				return RootHandle.Element;
			else
				return null;
		}

		protected override void ActiveChanged()
		{
			MarkSubscriptionOutOfDate();
			ElementChanged();
			base.ActiveChanged();
		}

		protected abstract void DeriveAndSubscribe(ISubscriber valueChangeSubscriber, ISubscriber resubscribeSubscriber);
		protected void MarkSubscriptionOutOfDate()
		{
			deriver.MarkSubscriptionOutOfDate();
			InternalElement = null;
		}

		protected override void EnsureInternalElement()
		{
			deriver.EnsureCurrent();
		}

		private ActiveChangedAdapter m_ActiveChangedAdapter;
		private VariableAdder variableAdder;

		protected IStaticContext EffectiveRootStaticContext
		{
			get { return variableAdder; }
		}

		protected RootedHandle(): base()
		{
			deriver = new RootedHandleDeriver(this);
			variableAdder = new VariableAdder(this);
		}

		protected sealed override IEcoTypeSystem GetTypeSystem()
		{
			return EffectiveRootStaticContext.TypeSystem;
		}
		/// <summary>
		/// For internal use, used by the designer.
		/// </summary>
		public bool IsRootLinkedTo(ElementHandle handle)
		{
			if (handle == RootHandle)
				return true;
			else if (RootHandle is RootedHandle)
				return (RootHandle as RootedHandle).IsRootLinkedTo(handle);
			else
				return false;
		}

		protected override bool RefersToComponent(object component)
		{
			bool Result = base.RefersToComponent(component);
			ElementHandle elementHandle = component as ElementHandle;
			if (!Result && (elementHandle != null))
				Result = IsRootLinkedTo(elementHandle);
			return Result;
		}

		/// <summary>
		/// Set this property to the ElementHandle that should be used as the root when deriving Element.
		/// </summary>
		///<exception cref="EcoException">Thrown if a circular chain is created when setting the new roothandle.</exception>
		[LocalizableCategory(typeof(HandlesStringRes), "sCategoryConnections")]
		[LocalizableDescription(typeof(HandlesStringRes), "sPropertyRootHandle")]
		[TypeConverter(typeof(RootHandleConverter))]
		[Browsable(true)]
		[DefaultValue(null)]
		[Editor("Borland.Eco.Handles.Design.RootHandleEditor, Borland.Eco.Handles.Design", typeof(UITypeEditor))]
		public ElementHandle RootHandle
		{
			get { return rootHandle; }
			set
			{
				if (RootHandle != value)
				{
					if ((value == this) ||
						((value is RootedHandle) && (value as RootedHandle).IsRootLinkedTo(this)))
						throw new EcoException(HandlesStringRes.sCircularReference); // FIXME (sCircularReference, [ClassName, 'SetInternalRootHandle', Name, Value.Name]); // do not localize
					ElementHandle oldRootHandle = rootHandle;
					// Drop all subscriptions by dropping old adapters
					if (rootChangedAdapter != null)
						rootChangedAdapter.Deactivate();
					if (rootStaticContextChangedAdapter != null)
						rootStaticContextChangedAdapter.Deactivate();
					if (m_ActiveChangedAdapter != null)
						m_ActiveChangedAdapter.Deactivate();
					rootChangedAdapter = new RootElementChangedAdapter(this);
					rootStaticContextChangedAdapter = new RootStaticContextChangedAdapter(this);
					m_ActiveChangedAdapter = new ActiveChangedAdapter(this);
					rootHandle = value;
					if (RootHandle != null)
					{
						RootHandle.SubscribeToElement(rootChangedAdapter);
						RootHandle.SubscribeToStaticContext(rootStaticContextChangedAdapter);
						RootHandle.SubscribeToActive(m_ActiveChangedAdapter);
					}
					if ((oldRootHandle == null) || (value == null))
						ActiveChanged();
					RootStaticContextChanged();
				}
			}
		}

		private IExternalVariableList effectiveVariables;
		protected override IExternalVariableList EffectiveVariables()
		{
			if (effectiveVariables == null)
			{
				IStaticContext rootContext = RootHandle;
				if (addRootVariables && (RootHandle != null) && (rootContext.VariableList != null))
				{
				if (Variables == null)
					effectiveVariables = rootContext.VariableList;
				else
					effectiveVariables = Variables.OclVariableCollection; // fixme merge
				}
				else
				{
				if (Variables == null)
					effectiveVariables = null;
				else
					effectiveVariables = Variables.OclVariableCollection;
				}

			}
			return effectiveVariables;
		}

		private bool addRootVariables;

		/// <summary>
		/// In addition to the variables defined on this handle, also add the handles defined by the RootHandle if this is true.
		/// </summary>
		[LocalizableCategory(typeof(HandlesStringRes), "sCategoryOCL")]
		[LocalizableDescription(typeof(HandlesStringRes), "sPropertyAddRootVariables")]
		[Browsable(true)]
		[DefaultValue(false)]
		public bool AddRootVariables
		{
			get { return addRootVariables; }
			set
			{
				if (AddRootVariables != value)
				{
					addRootVariables = value;
					StaticContextChanged();
				}
			}
		}

		public override bool Active
		{
			get { return Enabled && (RootHandle != null) && (RootHandle.Active); }
		}
	}
}